<?php
/**
 * Banyo Custom Checkout
 *
 * @category  Banyo
 * @package   Banyo\CustomCheckout
 */
declare(strict_types=1);

namespace Banyo\CustomCheckout\ViewModel;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Catalog\Helper\ImageFactory;

class OrderSuccess implements ArgumentInterface
{
    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var ImageFactory
     */
    private $imageHelperFactory;

    /**
     * @param CheckoutSession $checkoutSession
     * @param OrderRepositoryInterface $orderRepository
     * @param ImageFactory $imageHelperFactory
     */
    public function __construct(
        CheckoutSession $checkoutSession,
        OrderRepositoryInterface $orderRepository,
        ImageFactory $imageHelperFactory
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->orderRepository = $orderRepository;
        $this->imageHelperFactory = $imageHelperFactory;
    }

    /**
     * Get last order
     *
     * @return Order
     */
    public function getOrder(): Order
    {
        return $this->checkoutSession->getLastRealOrder();
    }

    /**
     * Get order items
     *
     * @return array
     */
    public function getOrderItems(): array
    {
        return $this->getOrder()->getAllVisibleItems();
    }

    /**
     * Get shipping address
     *
     * @return \Magento\Sales\Model\Order\Address|null
     */
    public function getShippingAddress()
    {
        return $this->getOrder()->getShippingAddress();
    }

    /**
     * Get billing address
     *
     * @return \Magento\Sales\Model\Order\Address|null
     */
    public function getBillingAddress()
    {
        return $this->getOrder()->getBillingAddress();
    }

    /**
     * Get payment method
     *
     * @return \Magento\Payment\Model\MethodInterface
     */
    public function getPaymentMethod()
    {
        return $this->getOrder()->getPayment()->getMethodInstance();
    }

    /**
     * Format price
     *
     * @param float|string $price
     * @return string
     */
    public function formatPrice($price): string
    {
        // Convert to float if string is provided
        if (is_string($price)) {
            $price = (float) $price;
        }
        return $this->getOrder()->formatPrice($price);
    }

    /**
     * Get product image URL for an order item
     *
     * @param \Magento\Sales\Model\Order\Item $item
     * @param string $imageType
     * @return string
     */
    public function getItemImageUrl($item, string $imageType = 'product_small_image'): string
    {
        $product = $item->getProduct();
        if (!$product) {
            // If product is not available, return empty string
            return '';
        }

        return $this->imageHelperFactory->create()->init($product, $imageType)->getUrl();
    }
}
