<?php
namespace Banyo\CustomHomePage\ViewModel;

use Magento\Catalog\Model\CategoryFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Store\Model\ScopeInterface;
use Magento\Catalog\Helper\ImageFactory;
use Magento\Review\Model\Review\SummaryFactory;

class BanyoOffers implements ArgumentInterface
{
    private CategoryFactory $categoryFactory;
    private ProductCollectionFactory $productCollectionFactory;
    private ScopeConfigInterface $scopeConfig;
    private StoreManagerInterface $storeManager;
    private ImageFactory $imageHelperFactory;
    private SummaryFactory $reviewSummaryFactory;

    public function __construct(
        CategoryFactory $categoryFactory,
        ProductCollectionFactory $productCollectionFactory,
        ScopeConfigInterface $scopeConfig,
        StoreManagerInterface $storeManager,
        ImageFactory $imageHelperFactory,
        SummaryFactory $reviewSummaryFactory
    ) {
        $this->categoryFactory = $categoryFactory;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->imageHelperFactory = $imageHelperFactory;
        $this->reviewSummaryFactory = $reviewSummaryFactory;
    }

    public function getProductImageUrl(\Magento\Catalog\Model\Product $product, string $imageType = 'product_base_image'): string
    {
        return $this->imageHelperFactory->create()->init($product, $imageType)->getUrl();
    }


    private function getCategoryByName(string $name)
    {
        $collection = $this->categoryFactory->create()
            ->getCollection()
            ->addAttributeToSelect('*')
            ->addAttributeToFilter('name', $name)
            ->setPageSize(1);

        return $collection->getFirstItem();
    }

    public function getBanyoOffersData(): array
    {
        return $this->getCategoryProductsData('Banyo Offers', 6, 'banyo_theme/banyo_offers_homepage_image/banyo_offers_image');
    }

    /**
     * Get products data for a specific category
     *
     * @param string $categoryName The name of the category to fetch products from
     * @param int $pageSize Number of products to fetch
     * @param string|null $imageConfigPath Path to the image configuration
     * @return array
     */
    public function getCategoryProductsData(string $categoryName, int $pageSize = 6, ?string $imageConfigPath = null): array
    {
        $category = $this->getCategoryByName($categoryName);

        if (!$category || !$category->getId()) {
            return ['products' => [], 'url' => '', 'name' => '', 'image' => ''];
        }

        $storeId = $this->storeManager->getStore()->getId();

        $products = $this->productCollectionFactory->create()
            ->addAttributeToSelect('*')
            ->addCategoryFilter($category)
            ->addAttributeToFilter('status', Status::STATUS_ENABLED)
            ->addAttributeToFilter('visibility', ['neq' => \Magento\Catalog\Model\Product\Visibility::VISIBILITY_NOT_VISIBLE])
            ->setStoreId($storeId)
            ->setPageSize($pageSize)
            ->setCurPage(1);

        $imageUrl = $imageConfigPath ? $this->getSlideImageUrl($imageConfigPath) : null;

        return [
            'products' => $products,
            'url' => $category->getUrl(),
            'name' => $category->getName(),
            'image' => $imageUrl
        ];
    }

    public function getSlideImageUrl(string $configPath): ?string
    {
        $value = $this->scopeConfig->getValue($configPath, ScopeInterface::SCOPE_STORE);
        if (!$value) return null;

        $mediaUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
        return $mediaUrl . 'wysiwyg/' . ltrim($value, '/');
    }

    /**
     * Get product rating summary
     *
     * @param int $productId
     * @return int
     */
    public function getProductRatingSummary($productId): int
    {
        $reviewSummary = $this->reviewSummaryFactory->create();
        $reviewSummary->setData('store_id', $this->storeManager->getStore()->getId());
        $reviewSummary->load($productId);
        return $reviewSummary->getRatingSummary() ?: 0;
    }
}
