<?php
/**
 * @category Mageants SimpleConfigurable
 * @package Mageants_SimpleConfigurable
 * @copyright Copyright (c) 2017 Mageants
 * @author Mageants Team <info@mageants.com>
 */

namespace Mageants\SimpleConfigurable\Plugin\Block\ConfigurableProduct\Cart\Item\Renderer;

use Magento\ConfigurableProduct\Block\Cart\Item\Renderer\Configurable as ConfigurableRenderer;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable as ConfigurableType;
use Mageants\SimpleConfigurable\Helper\Data as SCPHelper;
use Magento\Framework\App\RequestInterface;

/**
 * Class HideConfigurableOptions
 * 
 * Plugin to hide configurable options for configurable products in the cart.
 */
class HideConfigurableOptions
{
    /**
     * @var ConfigurableType
     */
    private $configurableType;

    /**
     * @var SCPHelper
     */
    protected $scpHelper;

    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * Constructor
     *
     * @param ConfigurableType $configurableType
     * @param SCPHelper $scpHelper
     * @param RequestInterface $request
     */
    public function __construct(
        ConfigurableType $configurableType,
        SCPHelper $scpHelper,
        RequestInterface $request
    ) {
        $this->configurableType = $configurableType;
        $this->scpHelper = $scpHelper;
        $this->request = $request;
    }

    /**
     * After Plugin for getOptionList
     * 
     * Filters out configurable options from the cart item’s option list.
     *
     * @param ConfigurableRenderer $subject
     * @param array $result Original options list
     * @return array Filtered options list without configurable attributes
     */
    public function afterGetOptionList(ConfigurableRenderer $subject, array $result)
    {
        $isEnable = ($this->scpHelper->isEnable() && $this->scpHelper->updateChieldInCart());
        if ($isEnable && in_array($this->request->getFullActionName(), ['checkout_cart_index'])) {
            $product = $subject->getItem()->getProduct();
            return $this->filterConfigurableAttributes($product, $result);
        } else {
            return $result;
        }
    }

    /**
     * Filters out configurable attributes from options.
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $options
     * @return array
     */
    protected function filterConfigurableAttributes($product, $options)
    {
        if ($product->getTypeId() !== 'configurable') {
            return $options;
        }

        try {
            $configurableAttributes = $this->configurableType->getConfigurableAttributesAsArray($product);
            $configurableAttributeLabels = array_column($configurableAttributes, 'label');

            $filteredOptions = array_values(array_filter($options, function ($option) use ($configurableAttributeLabels) {
                return !in_array($option['label'], $configurableAttributeLabels);
            }));

            return $filteredOptions;
        } catch (\Exception $e) {
            return $options;
        }
    }
}
